package main

import (
	"bytes"
	"encoding/hex"
	"flag"
	"fmt"
	"io"
	"os"
	. "speistel/pkg"
)

func fprintf(format string, a ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, format, a...)
}

func main() {
	// 1. parse cmd args
	var (
		inputFile  string
		inputHex   string
		outputFile string
		keyHex     string
	)

	flag.StringVar(&inputFile, "f", "", "input ciphertext file")
	flag.StringVar(&inputHex, "x", "", "input ciphertext as hex string")
	flag.StringVar(&outputFile, "o", "", "output file for decrypted plaintext (default: stdout)")
	flag.StringVar(&keyHex, "k", "", "hex-encoded key")
	flag.Parse()

	if inputFile == "" && inputHex == "" {
		fmt.Println("Error: provide either -f <file> or -x <hex_string>")
		os.Exit(1)
	}

	if keyHex == "" {
		fmt.Println("Error: provide hex-encoded key using -k <key_hex_string>")
		os.Exit(1)
	}
	if len(keyHex) != 64 {
		fmt.Println("Error: the hex-encoded key must be exactly 64 bytes long")
		os.Exit(1)
	}

	// 2. read ciphertext from either input string or file
	var ciphertext []byte
	if inputFile != "" {
		data, err := os.ReadFile(inputFile)
		if err != nil {
			fprintf("failed to read input file: %v\n", err)
			os.Exit(1)
		}
		ciphertext = data
	} else {
		data, err := hex.DecodeString(inputHex)
		if err != nil {
			fprintf("failed to decode hex string: %v\n", err)
			os.Exit(1)
		}
		ciphertext = data
	}

	// 4. decrypt the ciphertext
	plaintext, err := DecryptData(ciphertext, keyHex)
	if err != nil {
		fprintf("failed to decrypt ciphertext: %v\n", err)
		os.Exit(1)
	}

	// 5. save the decrypted data to a file or output to stdout
	if outputFile != "" {
		if err := os.WriteFile(outputFile, plaintext, 0644); err != nil {
			fprintf("failed to write output file: %v\n", err)
			os.Exit(1)
		}
		fmt.Printf("Decrypted data saved to %s\n", outputFile)
	} else {
		// print as raw bytes to stdout
		if _, err := io.Copy(os.Stdout, bytes.NewReader(plaintext)); err != nil {
			fprintf("failed to write plaintext to stdout: %v\n", err)
			os.Exit(1)
		}
	}
}
