package main

import (
	"encoding/binary"
	"flag"
	"fmt"
	"io"
	"net"
	"os"
)

func fprintf(format string, a ...interface{}) {
	_, _ = fmt.Fprintf(os.Stderr, format, a...)
}

func main() {
	// 1. parse cmd args
	var (
		addr      string
		inputStr  string
		inputFile string
		output    string
	)

	flag.StringVar(&addr, "addr", "127.0.0.1:8888", "server addr:port")
	flag.StringVar(&inputStr, "s", "", "input string to encrypt")
	flag.StringVar(&inputFile, "f", "", "input file to encrypt")
	flag.StringVar(&output, "o", "", "file to save the received ciphertext")
	flag.Parse()

	if inputStr == "" && inputFile == "" {
		fmt.Println("Error: provide either -s <input_string> or -f <input_file>")
		os.Exit(1)
	}

	// 2. read the input string or file
	var plaintext []byte
	if inputStr != "" {
		plaintext = []byte(inputStr)
	} else {
		data, err := os.ReadFile(inputFile)
		if err != nil {
			fprintf("failed to read input file: %v\n", err)
			os.Exit(1)
		}
		plaintext = data
	}

	// 3. encrypt using the Oracle
	// connect to the server
	conn, err := net.Dial("tcp", addr)
	if err != nil {
		fprintf("failed to connect: %v\n", err)
		os.Exit(1)
	}
	defer func(conn net.Conn) {
		err := conn.Close()
		if err != nil {
			fprintf("failed to close connection: %v\n", err)
		}
	}(conn)

	// send <length>+<plaintext> payload
	lenBuf := make([]byte, 4)
	binary.BigEndian.PutUint32(lenBuf, uint32(len(plaintext)))
	if _, err := conn.Write(lenBuf); err != nil {
		fprintf("failed to send length: %v\n", err)
		os.Exit(1)
	}
	if _, err := conn.Write(plaintext); err != nil {
		fprintf("failed to send data: %v\n", err)
		os.Exit(1)
	}

	// 4. recv the ciphertext
	if _, err := io.ReadFull(conn, lenBuf); err != nil {
		fprintf("failed to read ciphertext length: %v\n", err)
		os.Exit(1)
	}
	cipherLen := binary.BigEndian.Uint32(lenBuf)
	ciphertext := make([]byte, cipherLen)
	if _, err := io.ReadFull(conn, ciphertext); err != nil {
		fprintf("failed to read ciphertext: %v\n", err)
		os.Exit(1)
	}

	// 5. save the ciphertext to a file or output to stdout
	if output != "" {
		if err := os.WriteFile(output, ciphertext, 0644); err != nil {
			fprintf("failed to write output file: %v\n", err)
			os.Exit(1)
		}
		fmt.Printf("Ciphertext saved to %s\n", output)
	} else {
		// output as hex
		fmt.Printf("Ciphertext (%d bytes): %x\n", len(ciphertext), ciphertext)
	}
}
